package furny.swing.admin.tags;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.AbstractAction;
import javax.swing.AbstractListModel;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.TransferHandler;
import javax.swing.tree.TreePath;

import org.jdesktop.swingx.JXList;
import org.jdesktop.swingx.renderer.DefaultListRenderer;

import furny.entities.Tag;
import furny.entities.TagType;
import furny.furndb.FurnDBManager;
import furny.swing.admin.tags.TagTree.TagNode;
import furny.swing.admin.tags.TagTree.TagTypeNode;

/**
 * Panel for editing the tags of a furniture.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class EditFurnitureTagsPanel extends JPanel {

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(EditFurnitureTagsPanel.class.getName());

  private final List<Tag> tagsToAdd = new ArrayList<Tag>();

  private final JXList tagList;

  private final TagTree tagTree;

  /**
   * Instantiates a new edits the furniture tags panel.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public EditFurnitureTagsPanel() {
    setLayout(new GridBagLayout());
    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(10, 10, 10, 5);
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.gridheight = 2;
    constraints.fill = GridBagConstraints.BOTH;

    tagTree = new TagTree();
    tagTree.addMouseListener(new AddTagMouseListener());

    JScrollPane scrollPane = new JScrollPane(tagTree);
    scrollPane.setPreferredSize(new Dimension(150, 500));
    add(scrollPane, constraints);

    constraints.insets = new Insets(10, 5, 10, 5);
    constraints.gridx++;
    constraints.weightx = 0d;
    constraints.weighty = 1d;
    constraints.gridheight = 1;
    constraints.fill = GridBagConstraints.NONE;
    constraints.anchor = GridBagConstraints.SOUTH;

    add(new JButton(new ActionAddTag()), constraints);

    constraints.gridy++;
    constraints.anchor = GridBagConstraints.NORTH;

    add(new JButton(new ActionRemoveTag()), constraints);

    constraints.insets = new Insets(10, 5, 10, 10);
    constraints.gridy = 0;
    constraints.gridx++;
    constraints.weightx = 1.0d;
    constraints.weighty = 1.0d;
    constraints.gridheight = 2;
    constraints.fill = GridBagConstraints.BOTH;

    tagList = new JXList(new TagListModel());
    tagList.setTransferHandler(new TagTransferHandler());
    tagList.setCellRenderer(new TagListRenderer());
    tagList.setDragEnabled(true);
    tagList.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    tagList.addMouseListener(new RemoveTagMouseListener());

    scrollPane = new JScrollPane(tagList);
    scrollPane.setPreferredSize(new Dimension(150, 500));
    add(scrollPane, constraints);

    constraints.insets = new Insets(10, 5, 10, 10);
    constraints.gridx = 0;
    constraints.gridy += 2;
    constraints.weightx = 0.0d;
    constraints.weighty = 0.0d;
    constraints.gridheight = 2;
    constraints.fill = GridBagConstraints.NONE;
    constraints.anchor = GridBagConstraints.WEST;

    add(new JButton(new ActionCreateTag()), constraints);
  }

  /**
   * Gets the selected tags.
   * 
   * @return the selected tags
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public List<Tag> getSelectedTags() {
    return tagsToAdd;
  }

  /**
   * Adds tags.
   * 
   * @param tags
   *          Tags to add.
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void addTags(final List<Tag> tags) {
    for (final Tag tag : tags) {
      if (!tagsToAdd.contains(tag)) {
        tagsToAdd.add(tag);
      }
    }

    Collections.sort(tagsToAdd);
  }

  /**
   * Removes tags.
   * 
   * @param tags
   *          Tags to remove.
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void removeTags(final List<Tag> tags) {
    for (final Tag tag : tags) {
      tagsToAdd.remove(tag);
    }
  }

  /**
   * Action to remove tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionRemoveTag extends AbstractAction {

    /**
     * Instantiates a new action to remove tags.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionRemoveTag() {
      super("<<");
      putValue(SHORT_DESCRIPTION, "Remove the selected tags from the list");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      tagsToAdd.removeAll(Arrays.asList(tagList.getSelectedValues()));

      tagList.clearSelection();

      SwingUtilities.invokeLater(new Runnable() {
        @Override
        public void run() {
          ((TagListModel) tagList.getModel()).fireListChanged();
        }
      });

    }
  }

  /**
   * Mouse listener to add tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class AddTagMouseListener extends MouseAdapter {
    @Override
    public void mouseClicked(final MouseEvent e) {
      if (e.getButton() == MouseEvent.BUTTON1 && e.getClickCount() == 2) {
        addTags(tagTree.getSelectedTags());

        Collections.sort(tagsToAdd);
        ((TagListModel) tagList.getModel()).fireListChanged();
      }
    }
  }

  /**
   * Mouse listener to remove tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class RemoveTagMouseListener extends MouseAdapter {
    @Override
    public void mouseClicked(final MouseEvent e) {
      if (e.getButton() == MouseEvent.BUTTON1 && e.getClickCount() == 2) {
        tagsToAdd.removeAll(Arrays.asList(tagList.getSelectedValues()));

        tagList.clearSelection();

        SwingUtilities.invokeLater(new Runnable() {
          @Override
          public void run() {
            ((TagListModel) tagList.getModel()).fireListChanged();
          }
        });
      }
    }
  }

  /**
   * Action to add a tag.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionAddTag extends AbstractAction {

    /**
     * Instantiates a new action to add a tag.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionAddTag() {
      super(">>");
      putValue(SHORT_DESCRIPTION, "Add the selected tags to the list");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      addTags(tagTree.getSelectedTags());

      Collections.sort(tagsToAdd);
      ((TagListModel) tagList.getModel()).fireListChanged();
    }
  }

  /**
   * Action to create a new tag.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ActionCreateTag extends AbstractAction {

    /**
     * Instantiates a new action to create a tag.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ActionCreateTag() {
      super("Create Tag...");
      putValue(SHORT_DESCRIPTION, "Create a new tag");
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
      // try to get the seleted TagType
      final TreePath tp = tagTree.getSelectionModel().getSelectionPath();
      Object o = null;
      if (tp != null) {
        o = tp.getLastPathComponent();
      }

      TagType type = null;
      if (o instanceof TagNode) {
        type = ((TagNode) o).getTag().getType();
      } else if (o instanceof TagTypeNode) {
        type = ((TagTypeNode) o).getType();
      }

      // create tag creation panel with selected TagType (may be null)
      final TagCreationPane tcp = new TagCreationPane(type);
      final int n = JOptionPane.showConfirmDialog(null, tcp, "Create Tag",
          JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);

      if (n == JOptionPane.OK_OPTION) {
        final Tag tag = tcp.getTag();
        if (tag == null) {
          JOptionPane.showMessageDialog(EditFurnitureTagsPanel.this,
              "Tag cannot be created without a name", "Error",
              JOptionPane.ERROR_MESSAGE);
          return;
        } else {
          for (final Tag t : tagTree.getAllTags()) {
            if (t.equals(tag)) {
              JOptionPane
                  .showMessageDialog(EditFurnitureTagsPanel.this,
                      "This tag already exists", "Error",
                      JOptionPane.ERROR_MESSAGE);
              return;
            }
          }

          FurnDBManager.getInstance().saveTag(tag);
          tagTree.update();
        }
      }
    }
  }

  /**
   * List model for tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class TagListModel extends AbstractListModel {

    /**
     * Fire a list changed event.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public void fireListChanged() {
      fireContentsChanged(this, 0, Math.min(tagsToAdd.size() - 1, 0));
    }

    @Override
    public Object getElementAt(final int index) {
      if (index < getSize()) {
        return tagsToAdd.get(index);
      }
      return null;
    }

    @Override
    public int getSize() {
      return tagsToAdd.size();
    }
  }

  /**
   * List renderer for tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class TagListRenderer extends DefaultListRenderer {
    @Override
    public Component getListCellRendererComponent(final JList list,
        final Object value, final int index, final boolean isSelected,
        final boolean cellHasFocus) {
      final Tag t = (Tag) value;

      return super.getListCellRendererComponent(list, t.toString(), index,
          isSelected, cellHasFocus);
    }
  }

  /**
   * Transfer handler for tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class TagTransferHandler extends TransferHandler {

    /**
     * Instantiates a new tag transfer handler.
     * 
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public TagTransferHandler() {
    }

    @Override
    protected Transferable createTransferable(final JComponent component) {
      final List<Tag> selection = new ArrayList<Tag>();

      for (final Object o : tagList.getSelectedValues()) {
        if (o instanceof Tag) {
          selection.add((Tag) o);
        }
      }

      tagsToAdd.removeAll(selection);

      ((TagListModel) tagList.getModel()).fireListChanged();

      if (!selection.isEmpty()) {

        return new TagTransferable(selection);
      } else {
        return null;
      }
    }

    @Override
    public boolean canImport(final JComponent comp,
        final DataFlavor[] transferFlavors) {
      for (final DataFlavor df : transferFlavors) {
        if (df.equals(new TagFlavor())) {
          return true;
        }
      }
      return false;
    }

    @SuppressWarnings("unchecked")
    @Override
    public boolean importData(final JComponent comp, final Transferable trans) {

      try {
        final List<Tag> tags = (List<Tag>) trans
            .getTransferData(new TagFlavor());

        addTags(tags);
        ((TagListModel) tagList.getModel()).fireListChanged();

        return true;
      } catch (final Exception e) {
        LOGGER.log(Level.SEVERE, "Failed to add tags", e);
      }

      return false;
    }

    @Override
    public int getSourceActions(final JComponent c) {
      return MOVE;
    }
  }

  /**
   * Panel for creating new tags.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private static class TagCreationPane extends JPanel {
    private final TagComboBox comboBox;
    private final JTextField tagNameField;

    /**
     * Instantiates a new tag creation pane.
     * 
     * @param type
     *          the type of the tags.
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public TagCreationPane(final TagType type) {
      setLayout(new GridBagLayout());
      final GridBagConstraints constraints = new GridBagConstraints();
      constraints.insets = new Insets(10, 10, 10, 5);
      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.weightx = 0.0d;
      constraints.weighty = 0.0d;
      constraints.fill = GridBagConstraints.NONE;

      comboBox = new TagComboBox();

      if (type != null) {
        SwingUtilities.invokeLater(new Runnable() {
          @Override
          public void run() {
            comboBox.setSelectedItem(type);

            SwingUtilities.invokeLater(new Runnable() {
              @Override
              public void run() {
                tagNameField.requestFocusInWindow();
              }
            });
          }
        });
      }

      add(comboBox, constraints);

      constraints.gridx++;
      tagNameField = new JTextField(25);
      add(tagNameField, constraints);
    }

    /**
     * Gets the tag.
     * 
     * @return the tag
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public Tag getTag() {
      final String name = tagNameField.getText();
      if (name != null && !name.isEmpty()) {
        return new Tag(comboBox.getSelectedTagType(), name);
      }

      return null;
    }
  }
}
